import { v2 as cloudinary } from "cloudinary";
import sendEmail from "../utils/sendEmail.js";

export const submitResume = async (req, res) => {
  try {
    if (!req.files || !req.files.resume) {
      return res.status(400).json({ message: "Resume file is required" });
    }
    const { resume } = req.files;
    const allowedFormats = [
      "application/pdf",
      "application/msword",
      "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
    ];

    if (!allowedFormats.includes(resume.mimetype)) {
      return res
        .status(400)
        .json({ message: "Only PDF & Word files are allowed" });
    }

    const { fullName, phone, email, position } = req.body;

    if (!fullName || !email || !phone || !position) {
      return res.status(400).json({ message: "All fields are required" });
    }

    const cloudinaryResponse = await cloudinary.uploader.upload(
      resume.tempFilePath,
      {
        resource_type: "raw",
        folder: "RESUMES",
      }
    );

    if (!cloudinaryResponse || cloudinaryResponse.error) {
      return res
        .status(500)
        .json({ message: "Failed to upload resume to Cloudinary" });
    }

    const submissionDate = new Date().toLocaleString("en-US", {
      timeZone: "Asia/Kolkata",
      year: "numeric",
      month: "long",
      day: "numeric",
      hour: "2-digit",
      minute: "2-digit",
      second: "2-digit",
    });

    const emailSubject = `${fullName}`;
    const emailContent = `
     <div style="font-family: Arial, sans-serif; max-width: 640px; margin: 0 auto; border: 1px solid #ddd; border-radius: 8px; overflow: hidden;">
      <div style="background-color: #be1966; padding: 16px; text-align: center; color: white;">
        <h2 style="margin: 0; font-size: 20px;">New Resume Submission</h2>
      </div>

      <div style="background-color: #ffffff; padding: 24px;">
        <p style="font-size: 16px; margin-bottom: 10px;"><strong>Name: </strong>${fullName}</p>
        <p style="font-size: 16px; margin-bottom: 10px;"><strong>Phone: </strong>${phone}</p>
        <p style="font-size: 16px; margin-bottom: 10px;"><strong>Email: </strong>${email}</p>
        <p style="font-size: 16px; margin-bottom: 10px;"><strong>Role Applied For: </strong>${position}</p>
        <p style="font-size: 16px; margin-bottom: 10px;">
          <strong>Resume:</strong>
          <a href="${cloudinaryResponse.secure_url}" target="_blank" style="color: #0073e6; text-decoration: none;">
            Download Resume
          </a>
        </p>
        <p style="font-size: 16px; margin-bottom: 0;"><strong>Submitted On: </strong>${submissionDate}</p>
      </div>
    </div>

    `;

    await sendEmail(
      "contact@sohiljain.com",
      emailSubject,
      emailContent,
      "Ska Alliance Resume"
    );

    res.status(200).json({ message: "Resume Sent Successfully!" });
  } catch (error) {
    console.error("Error:", error.message);
    res.status(500).json({ message: "Internal server error" });
  }
};
